<?php

namespace App\Custom;

use App\Models\Operation;
use App\Models\ProductPurchase;
use App\Models\ProductPurchaseItem;
use App\Models\TattooArtist;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DashboardCustom
{
    public static function getTodayIncoming($id)
    {
        $incoming = Operation::where('studioObjectId', $id)
            ->whereDate('date', date('Y-m-d'))
            ->where('saleObjectId', '!=', null)
            ->sum('amount');

        $advances = Operation::join('sales', 'sales.objectId', '=', 'operations.saleObjectId')
            ->where('operations.studioObjectId', $id)
            ->whereDate('operations.date', date('Y-m-d'))
            ->where('operations.saleObjectId', '!=', null)
            ->sum('sales.advance');

        $grossIncoming = $incoming + $advances;

        // Métodos de pago
        $paymentMethodTypes = config('app.paymentMethodTypes');
        $paymentMethods = DB::table('operations')->join('sales', 'sales.objectId', '=', 'operations.saleObjectId')
            ->join('payment_methods', 'payment_methods.objectId', '=', 'sales.paymentMethodObjectId')
            ->where('operations.studioObjectId', $id)
            ->whereDate('operations.date', date('Y-m-d'))
            ->groupBy('payment_methods.typeId')
            ->orderBy(DB::raw('SUM(operations.amount + sales.advance)'), 'DESC')
            ->select(
                'payment_methods.typeId',
                DB::raw('SUM(operations.amount) AS amount'),
                DB::raw('SUM(sales.advance) AS advance'),
                DB::raw('SUM(operations.amount + sales.advance) AS total')
            )->get();


        for ($i = 0; $i < count($paymentMethods); ++$i) {
            $paymentMethod = $paymentMethods[$i];
            $paymentMethods[$i]->total = "$" . number_format($paymentMethods[$i]->total, 2);
            $paymentMethods[$i]->typeName = $paymentMethodTypes[$paymentMethod->typeId];
        }

        Log::debug($paymentMethods);

        $products = ProductPurchaseItem::join('product_purchases', 'product_purchases.objectId', '=', 'product_purchase_items.productPurchaseObjectId')
            ->join('tattoo_artists', 'tattoo_artists.objectId', '=', 'product_purchases.tattooArtistObjectId')
            ->where('tattoo_artists.studioObjectId', $id)
            ->where('product_purchase_items.isCancelled', false)
            ->where('product_purchase_items.isDeleted', false)
            ->whereDate('product_purchase_items.created_at', date('Y-m-d'))
            ->get();

        $productsIncoming = ProductPurchaseItem::join('product_purchases', 'product_purchases.objectId', '=', 'product_purchase_items.productPurchaseObjectId')
            ->join('tattoo_artists', 'tattoo_artists.objectId', '=', 'product_purchases.tattooArtistObjectId')
            ->where('tattoo_artists.studioObjectId', $id)
            ->where('product_purchase_items.isCancelled', false)
            ->where('product_purchase_items.isDeleted', false)
            ->whereDate('product_purchase_items.created_at', date('Y-m-d'))
            ->select(DB::raw('SUM(product_purchase_items.quantity * product_purchase_items.price) AS incoming'))
            ->first()
            ->incoming ?? 0;

        $productsQuantity = ProductPurchaseItem::join('product_purchases', 'product_purchases.objectId', '=', 'product_purchase_items.productPurchaseObjectId')
            ->join('tattoo_artists', 'tattoo_artists.objectId', '=', 'product_purchases.tattooArtistObjectId')
            ->where('tattoo_artists.studioObjectId', $id)
            ->where('product_purchase_items.isCancelled', false)
            ->where('product_purchase_items.isDeleted', false)
            ->whereDate('product_purchase_items.created_at', date('Y-m-d'))
            ->sum('product_purchase_items.quantity'); // Cantidad total de productos vendidos
            // ->count(); // Cantidad total de productos diferentes vendidos (sin importar la cantidad)

        // $productsQuantity = ProductPurchase::join('tattoo_artists', 'tattoo_artists.objectId', '=', 'product_purchases.tattooArtistObjectId')
        //     ->where('tattoo_artists.studioObjectId', $id)
        //     ->whereDate('product_purchases.created_at', date('Y-m-d'))
        //     ->count(); // Cantidad de ventas de productos totales (sin importar si son productos diferentes ni la cantidad)

        return [
            'grossIncoming' => "$" . number_format($grossIncoming, 2),
            'incoming' => "$" . number_format($incoming, 2),
            'paymentMethods' => $paymentMethods,
            'products' => [
                'data' => $products,
                'quantity' => $productsQuantity,
                'grossIncoming' => "$" . number_format($productsIncoming, 2)
            ]
        ];
    }

    public static function getMonthlyIncoming($id)
    {
        $bottomLabels = [];
        $displayLabels = [];
        $data = [];

        $startDate = date('Y-m-01');
        $endDate = date('Y-m-d');

        // Get bottom labels
        $startTS = strtotime($startDate);
        $endTS = strtotime($endDate);
        $daySeconds = 86400;

        for ($i = $startTS; $i <= $endTS; $i += $daySeconds) {
            $bottomLabels[] = date('Y-m-d', $i);
            $displayLabels[] = date('d/m', $i);
        }

        // Get tattoos incoming
        $operations = Operation::join('sales', 'operations.saleObjectId', '=', 'sales.objectId')
            ->where('studioObjectId', $id)
            ->where('saleObjectId', '!=', null)
            ->whereDate('date', '>=', $startDate)
            ->whereDate('date', '<=', $endDate)
            ->groupBy(DB::raw("CAST(operations.date AS DATE)"))
            ->orderBy(DB::raw("CAST(operations.date AS DATE)"), 'ASC')
            ->select(
                DB::raw("CAST(operations.date AS DATE) AS date"),
                DB::raw('SUM(operations.amount) AS amount'),
                DB::raw('SUM(sales.advance) AS advance')
            )->get();

        $data[0] = [];
        for ($i = 0; $i < count($bottomLabels); ++$i) {
            $labelDate = strtotime($bottomLabels[$i]);
            $found = false;

            for ($j = 0; $j < count($operations); ++$j) {
                $operation = $operations[$j];
                $operationDate = strtotime($operation->date);

                if ($labelDate == $operationDate) {
                    $found = true;
                    $data[0]['label'] = 'Tatuajes';
                    $data[0]['backgroundColor'] = '#2a2a2a';
                    $data[0]['borderColor'] = '#2a2a2a';
                    $data[0]['data'][] = $operation->amount + $operation->advance;
                }
            }

            if (!$found) {
                $data[0]['label'] = 'Tatuajes';
                $data[0]['backgroundColor'] = '#2a2a2a';
                $data[0]['borderColor'] = '#2a2a2a';
                $data[0]['data'][] = 0;
            }

        }

        // Get products incoming
        $productPurchaseItems = ProductPurchaseItem::join('product_purchases', 'product_purchase_items.productPurchaseObjectId', '=', 'product_purchases.objectId')
            ->join('tattoo_artists', 'product_purchases.tattooArtistObjectId', '=', 'tattoo_artists.objectId')
            ->join('studios', 'tattoo_artists.studioObjectId', '=', 'studios.objectId')
            ->whereDate('product_purchase_items.created_at', '>=', $startDate)
            ->whereDate('product_purchase_items.created_at', '<=', $endDate)
            ->where('product_purchase_items.isCancelled', false)
            ->where('product_purchase_items.isDeleted', false)
            ->where('studios.objectId', $id)
            ->groupBy(DB::raw("CAST(product_purchase_items.created_at AS DATE)"))
            ->orderBy(DB::raw("CAST(product_purchase_items.created_at AS DATE)"), 'ASC')
            ->select(DB::raw("CAST(product_purchase_items.created_at AS DATE) AS date"), DB::raw('SUM(product_purchase_items.quantity * product_purchase_items.price) AS subtotal'))
            ->get();

        $data[1] = [];
        for ($i = 0; $i < count($bottomLabels); ++$i) {
            $found = false;
            $labelDate = strtotime($bottomLabels[$i]);

            for ($j = 0; $j < count($productPurchaseItems); ++$j) {
                $item = $productPurchaseItems[$j];
                $itemDate = strtotime($item->date);

                if ($itemDate == $labelDate) {
                    $found = true;
                    $data[1]['label'] = 'Productos';
                    $data[1]['backgroundColor'] = '#bbb';
                    $data[1]['borderColor'] = '#bbb';
                    $data[1]['data'][] = $item->subtotal;
                    break;
                }
            }

            if (!$found) {
                $data[1]['label'] = 'Productos';
                $data[1]['backgroundColor'] = '#e2e3e5';
                $data[1]['borderColor'] = '#e2e3e5';
                $data[1]['data'][] = 0;
            }
        }

        return [
            'labels' => $displayLabels,
            'datasets' => $data
        ];
    }

    public static function getProductMonthlyIncoming($id)
    {
        $labels = [];
        $data = [];
        $backgrounds = [];

        $startDate = date('Y-m-01');
        $endDate = date('Y-m-d');

        $productPurchaseItems = ProductPurchaseItem::join('product_purchases', 'product_purchase_items.productPurchaseObjectId', '=', 'product_purchases.objectId')
            ->join('tattoo_artists', 'product_purchases.tattooArtistObjectId', '=', 'tattoo_artists.objectId')
            ->join('studios', 'tattoo_artists.studioObjectId', '=', 'studios.objectId')
            ->whereDate('product_purchase_items.created_at', '>=', $startDate)
            ->whereDate('product_purchase_items.created_at', '<=', $endDate)
            ->where('product_purchase_items.isCancelled', false)
            ->where('product_purchase_items.isDeleted', false)
            ->where('studios.objectId', $id)
            ->groupBy('product_purchase_items.productObjectId')
            ->select(
                'product_purchase_items.productObjectId',
                DB::raw('SUM(product_purchase_items.quantity * product_purchase_items.price) AS subtotal'),
                DB::raw('SUM(product_purchase_items.quantity) AS quantity')
            )
            ->get();

        $totalProducts = count($productPurchaseItems);
        for ($i = 0; $i < $totalProducts; ++$i) {
            $productPurchaseItem = $productPurchaseItems[$i];
            $labels[] = ProductPurchaseItem::select('name')
                ->where('productObjectId', $productPurchaseItem->productObjectId)
                ->first()
                ->name;
            $data[] = $productPurchaseItem->subtotal;

            $background = (1 / $totalProducts) * ($i + 1);
            $backgrounds[] = "rgba(0, 0, 0, $background)";
        }

        return [
            'productPurchases' => $productPurchaseItems,
            'labels' => $labels,
            'datasets' => [
                [
                    'data' => $data,
                    'backgroundColor' => $backgrounds
                ]
            ]
        ];
    }

    public static function getTattooArtistMonthlyIncoming($id)
    {
        $bottomLabels = [];
        $data = [];

        $startDate = date('Y-m-01');
        $endDate = date('Y-m-d');

        $operations = Operation::leftJoin('sales', 'sales.objectId', '=', 'operations.saleObjectId')
            ->leftJoin('advance_payments', 'advance_payments.objectId', '=', 'operations.advancePaymentObjectId')
            ->join('tattoo_artists', 'tattoo_artists.objectId', '=', 'operations.tattooArtistObjectId')
            ->join('users', 'users.objectId', '=', 'tattoo_artists.userObjectId')
            ->whereDate('date', '>=', $startDate)
            ->whereDate('date', '<=', $endDate)
            ->where('operations.studioObjectId', $id)
            ->where('tattoo_artists.isDeleted', false)
            ->where('users.isDeleted', false)
            ->groupBy('operations.tattooArtistObjectId')
            ->select(
                'operations.tattooArtistObjectId',
                DB::raw('SUM(sales.salePercentage * operations.amount) AS amount'),
                DB::raw("SUM(CASE WHEN operations.advancePaymentObjectId = '' THEN operations.amount ELSE 0 END) AS advance"),
                DB::raw("SUM(CASE WHEN operations.advancePaymentObjectId <> '' THEN operations.amount ELSE 0 END) AS advancePaymentAmount")
            )->get();

        $tattooArtistData = [];
        for ($i = 0; $i < count($operations); ++$i) {
            $operation = $operations[$i];
            $tattooArtist = TattooArtist::find($operation->tattooArtistObjectId);
            $bottomLabels[] =  $tattooArtist->user->name;
            $tattooArtistData[] = $operation->amount - $operation->advancePaymentAmount;
       }

       $data = [
            'label' => 'Tatuadores',
            'data' => $tattooArtistData,
            'backgroundColor' => '#2A2A2A',
            'borderColor' => '#2A2A2A'
       ];

        return [
            'labels' => $bottomLabels,
            'datasets' => [$data]
        ];
    }

    public static function groupByDate($data, $field) {
        $result = [];
        if ($data && count($data) > 0) {
            $i = 0;
            foreach($data as $d) {
                $date = date('Y-m-d', strtotime($d->{$field}));

                if (array_key_exists($i, $result)) {
                    if (array_key_exists($date, $result[$i])) {
                        $result[$i][$date][] = $d;
                    } else {
                        ++$i;
                        $result[$i] = [];
                        $result[$i][$date] = [];
                        $result[$i][$date][] = $d;
                        continue;
                    }
                } else {
                    $result[$i] = [];
                    $result[$i][$date] = [];
                    $result[$i][$date][] = $d;
                }
            }
        }

        return $result ?? [];
    }
}
