<?php

namespace App\Http\Controllers;

use App\Models\Business;
use App\Traits\FilesTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class BusinessController extends Controller
{
    use FilesTrait;

    public function index()
    {
        try {
            $businesses = Business::where('isDeleted', false)->get();
            return response()->json(['businesses' => $businesses], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function show($id)
    {
        try {
            $business = Business::where('isDeleted', false)->find($id);

            if (!$business) return response()->json(['error' => 'Bad URI'], 404);

            return response()->json(['business' => $business], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function store(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));

            $data = $request->all();
            unset($data['logo']);
            $business = Business::create($data);

            if ($request->hasFile('logo')) {
                $business->logo = $this->createFile('business', 'logo', $business->objectId, $request->logo);
                $business->save();
            }

            return response()->json(['message' => 'Estudio creado exitosamente'], 201);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $business = Business::where('isDeleted', false)->find($id);

            if (!$business) return response()->json(['error' => 'Bad URI'], 404);

            $this->validate($request, $this->validator($request));

            $data = $request->all();
            unset($data['logo']);
            $business->update($data);

            if ($request->hasFile('logo')) {
                $business->logo = $this->createFile('business', 'logo', $business->objectId, $request->logo);
                $business->save();
            }

            return response()->json([
                'message' => 'Negocio actualizado correctamente',
                'business' => Business::find($business->objectId)
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function destroy($id)
    {
        try {
            DB::beginTransaction();

            $business = Business::where('isDeleted', false)->find($id);
            if (!$business) return response()->json(['error' => 'Bad URI'], 404);

            $business->deleteBusiness();

            DB::commit();
            return response()->json(['message' => 'Estudio eliminado exitosamente'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    private function validator($request)
    {
        $max_file = env('UPLOAD_MAX_FILESIZE');

        $data = $request->all();
        $rules = [
            'name' => 'required',

            'address' => 'required|max:500',

            'phone' => 'required|digits:10|numeric',
            'website' => 'nullable',
            'email' => 'nullable|min:5|max:60|regex:/^.+@.+[.]+.+$/',
            'logo' => "nullable|max:$max_file",
            'legend' => 'nullable'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
