<?php

namespace App\Http\Controllers;

use App\Custom\FormCustom;
use App\Models\Admin;
use App\Models\Business;
use App\Models\DefaultQuestion;
use App\Models\Form;
use App\Models\FormResponse;
use App\Models\Question;
use App\Models\Response;
use App\Models\Studio;
use App\Models\TattooArtist;
use App\Models\User;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class FormController extends Controller
{
    public function index(Request $request) {
        try {

            // $validator = Validator::make(['questions' => $questions], $rules, $messages);
            // if ($validator->fails()) {
            //     return response()->json([
            //         'error'=> $validator->errors()->first()
            //     ], 400);
            // }

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', Auth::user()->businessObjectId)
                ->first();

            if (!$form) {
                $form = Form::create([
                    'businessObjectId' => Auth::user()->businessObjectId
                ]);

                $defaultQuestions = DefaultQuestion::where('isDeleted', false)
                    ->get();

                foreach($defaultQuestions as $defaultQuestion) {
                    Question::create([
                        'questionNumber' => $defaultQuestion->questionNumber,
                        'question' => $defaultQuestion->question,
                        'options' => $defaultQuestion->options,
                        'type' => $defaultQuestion->type,
                        'required' => $defaultQuestion->required,
                        'defaultQuestionObjectId' => $defaultQuestion->objectId,
                        'formObjectId' => $form->objectId
                    ]);
                }
            }

            $form = $request->has('questions')
                ? $this->formatForm($form)
                : $form;

            if ($request->has('defaultQuestions')) {
                $defaultQuestions = DefaultQuestion::where('isDeleted', false)
                    ->orderBy('questionNumber', 'ASC')
                    ->get();
                $form['defaultQuestions'] = $defaultQuestions;
            }

            if ($request->has('formResponses')) {
                if ($request->has('startDate') && $request->has('endDate')) {
                    $form['formResponses'] = $this->getFormResponses($request->startDate, $request->endDate);
                } else {
                    $end = date('Y-m-d');
                    $start = date('Y-m-d', strtotime("$end - 7 days"));
                    $form['formResponses'] = $this->getFormResponses(
                        $start, $end
                    );
                }
            }

            return response()->json([
                'form' => $form
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function formatForm($form) {
        $form = Form::find($form->objectId);
        $questions = Question::where('isDeleted', false)
            ->where('formObjectId', $form->objectId)
            ->orderBy('questionNumber', 'ASC')
            ->get();

        $form['questions'] = $questions;
        return $form;
    }

    public function groupBy($data) {
        $result = [];
        if ($data && count($data) > 0) {
            $i = 0;
            foreach($data as $d) {
                $date = date('Y-m-d', strtotime($d->created_at));

                if (array_key_exists($i, $result)) {
                    if (array_key_exists($date, $result[$i])) {
                        $result[$i][$date][] = $this->formatFormResponse($d);
                    } else {
                        ++$i;
                        $result[$i] = [];
                        $result[$i][$date] = [];
                        $result[$i][$date][] = $this->formatFormResponse($d);
                        continue;
                    }
                } else {
                    $result[$i] = [];
                    $result[$i][$date] = [];
                    $result[$i][$date][] = $this->formatFormResponse($d);
                }
            }
        }

        return $result ?? [];
    }

    public function formatFormResponse($formResponse, $transform = false) {
        $formResponse = FormResponse::find($formResponse->objectId);
        $responses = Response::where('isDeleted', false)
            ->where('formResponseObjectId', $formResponse->objectId)
            ->orderBy('questionNumber', 'ASC')
            ->get();

        if (!$transform) {
            foreach($responses as $key => $response) {
                if ($response->type != 'tattooArtist') continue;
                $tattooArtist = TattooArtist::find($response['data'][0]);
                if ($tattooArtist)
                    $responses[$key]['data'] = [$tattooArtist->user->name];
            }
        }

        $studio = Studio::find($formResponse->studioObjectId);

        $formResponse['responses'] = $responses;
        $formResponse['studio'] = $studio;

        return $formResponse;
    }

    public function getformForTattoo(Request $request) {
        try {

            $tattooArtist = Auth::user();
            $studio = Studio::where('isDeleted', false)
                ->find($tattooArtist->studioObjectId);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 404);

            $business = Business::where('isDeleted', false)
                ->find($studio->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->first();

            if (!$form) {
                $form = Form::create([
                    'businessObjectId' => Auth::user()->businessObjectId
                ]);

                $defaultQuestions = DefaultQuestion::where('isDeleted', false)
                    ->get();

                foreach($defaultQuestions as $defaultQuestion) {
                    Question::create([
                        'questionNumber' => $defaultQuestion->questionNumber,
                        'question' => $defaultQuestion->question,
                        'options' => $defaultQuestion->options,
                        'type' => $defaultQuestion->type,
                        'required' => $defaultQuestion->required,
                        'defaultQuestionObjectId' => $defaultQuestion->objectId,
                        'formObjectId' => $form->objectId
                    ]);
                }
            }

            $form = $this->formatForm($form);

            return response()->json([
                'form' => $form
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function getTodayFormResponses(Request $request) {
        try {
            $tattooArtist = Auth::user();
            $studio = Studio::where('isDeleted', false)
                ->find($tattooArtist->studioObjectId);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 404);

            $business = Business::where('isDeleted', false)
                ->find($studio->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->first();

            if (!$form) return response()->json([
                'error' => 'Formulario no válido.'
            ], 404);

            $formResponses = FormResponse::where('isDeleted', false)
                ->where('formObjectId', $form->objectId)
                ->where('studioObjectId', $studio->objectId)
                // ->where('finished', false)
                ->orderBy('created_at', 'DESC');

            $start = date('Y-m-d');
            $end = date('Y-m-d');
            $formResponses = $formResponses->whereDate('created_at', '>=', $start)
                ->whereDate('created_at', '<=', $end)
                ->get();

            foreach($formResponses as $key => $formResponse) {
                $formResponses[$key] = $this->formatFormResponse($formResponse, true);
            }

            $form = $this->formatForm($form);
            $form['formResponses'] = $formResponses;

            return response()->json([
                'form' => $form
            ], 200);
        } catch (ModelNotFoundException $e) {
            return null;
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function finishFormResponse(Request $request, $id) {
        try {
            $tattooArtist = Auth::user();
            $studio = Studio::where('isDeleted', false)
                ->find($tattooArtist->studioObjectId);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 404);

            $business = Business::where('isDeleted', false)
                ->find($studio->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->first();

            if (!$form) return response()->json([
                'error' => 'Formulario no válido.'
            ], 404);

            $formResponse = FormResponse::where('isDeleted', false)
                ->where('formObjectId', $form->objectId)
                // ->where('finished', false)
                ->find($id);

            if (!$formResponse) return response()->json([
                'error' => 'Respuestas no válidas.'
            ], 404);

            $formResponse->finished = true;
            $formResponse->save();

            return response()->json([
                'message' => 'Formulario finalizado.'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function getFormResponses($startDate = null, $endDate = null) {
        try {
            $admin = Auth::user();
            $business = Business::where('isDeleted', false)
                ->find($admin->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->first();

            if (!$form) return response()->json([
                'formResponses' => []
            ], 200);

            $formResponses = FormResponse::where('isDeleted', false)
                ->where('formObjectId', $form->objectId)
                ->orderBy('created_at', 'DESC');

            if ($startDate && $endDate) {
                $start = date('Y-m-d', strtotime($startDate));
                $end = date('Y-m-d', strtotime($endDate));
                $formResponses = $formResponses->whereDate('created_at', '>=', $start)
                    ->whereDate('created_at', '<=', $end)
                    ->get();
            }

            $groupedFormResponses = $this->groupBy($formResponses);
            // foreach($formResponses as $key => $formResponse) {
            //     $formResponses[$key] = $this->formatFormResponse($formResponse);
            // }

            return $groupedFormResponses;
        } catch (ModelNotFoundException $e) {
            return null;
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function show(Request $request, $id) {
        try {
            $form = Form::where('isDeleted', false)->find($id);

            return response()->json([
                'form' => $this->formatForm($form)
            ], 200);
        }catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function store(Request $request) {
        try {
            DB::beginTransaction();

            $json = $request->all();
            $questions = json_decode($json['questions'], true);

            $rules = [
                'questions' => 'required|array',
                'questions.*.question' => 'required|string',
                // 'questions.*.options' => 'sometimes|array|min:1',
                'questions.*.options.*' => 'required|string',
                'questions.*.type' => 'required|string',
                'questions.*.required' => 'required|boolean',
            ];
            $messages = [
                'questions.required' => 'El arreglo de preguntas es requerido.',
                'questions.array' => 'Las preguntas deben ser un arreglo.',
                'questions.*.question.required' => 'La pregunta es requerida.',
                'questions.*.question.string' => 'La pregunta debe ser una cadena de texto.',
                // 'questions.*.options.array' => 'Las opciones deben ser un arreglo.',
                // 'questions.*.options.min' => 'Agrega al menos una opción.',
                'questions.*.options.*.required' => 'La opción es requerida.',
                'questions.*.options.*.string' => 'La opción debe ser una cadena de texto.',
                'questions.*.type.required' => 'El tipo de campo es requerido.',
                'questions.*.type.string' => 'El tipo de campo debe ser una cadena de texto.',
                'questions.*.required.required' => 'El campo de obligatorio es requerido.',
                'questions.*.required.boolean' => 'El campo de obligatorio debe ser un booleano.',
            ];

            $validator = Validator::make(['questions' => $questions], $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            foreach($questions as $key => $question) {
                if (array_key_exists('isDefault', $question)) {
                    if (!array_key_exists('objectId', $question)) return response()->json([
                        'error' => 'Pregunta no válida.'
                    ], 400);

                    $defaultQuestion = DefaultQuestion::where('isDefault', true)
                        ->where('isDeleted', false)
                        ->find($question['objectId']);

                    if (!$defaultQuestion) return response()->json([
                        'error' => 'Pregunta no válida.'
                    ], 400);

                    $questions[$key] = [
                        'question' => $defaultQuestion->question,
                        'type' => $defaultQuestion->type,
                        'options' => $defaultQuestion->options,
                        'required' => $defaultQuestion->required,
                        'defaultQuestionObjectId' => $defaultQuestion->objectId,
                    ];
                    continue;
                }

                if (array_key_exists('defaultQuestionObjectId', $question) && $question['defaultQuestionObjectId']) {
                    $defaultQuestion = DefaultQuestion::where('isDefault', true)
                        ->where('isDeleted', false)
                        ->find($question['defaultQuestionObjectId']);

                    if (!$defaultQuestion) return response()->json([
                        'error' => 'Pregunta no válida.'
                    ], 406);

                    $questions[$key] = [
                        'question' => $defaultQuestion->question,
                        'type' => $defaultQuestion->type,
                        'options' => $defaultQuestion->options,
                        'required' => $defaultQuestion->required,
                        'defaultQuestionObjectId' => $defaultQuestion->objectId,
                    ];
                    continue;
                }

                $type = $question['type'];

                if (!in_array($type, FormCustom::FIELD_TYPES)) {
                    return response()->json([
                        'error' => 'Tipo de campo no válido.'
                    ], 400);
                }

                if ($type == FormCustom::CHECKBOX || $type == FormCustom::RADIO || $type == FormCustom::SELECT) {
                    if (!array_key_exists('options', $question)) {
                        return response()->json([
                            'error' => 'Las opciones son requeridas.'
                        ], 400);
                    }
                    if (is_array($question['options'])) {
                        if (count($question['options']) == 0)
                            return response()->json([
                                'error' => 'Las opciones son requeridas.'
                            ], 400);
                     }
                }
            }

            $admin = Auth::user();
            $business = Business::where('isDeleted', false)
                ->find($admin->businessObjectId);

            if (!$business) {
                return response()->json([
                    'error' => 'Negocio no válido.'
                ], 406);
            }

            // $form = Form::create([
            //     'businessObjectId' => $business->objectId
            // ]);

            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->first();

            $delQuestions = Question::where('isDeleted', false)
                ->where('formObjectId', $form->objectId)
                ->get();

            $delQuestions->each->update([
                'isDeleted' => true
            ]);

            $i = 0;
            foreach($questions as $question) {
                Question::create([
                    'questionNumber' => $i,
                    'question' => $question['question'],
                    'description' => array_key_exists('description', $question)
                        ? $question['description']
                        : null,
                    'options' => array_key_exists('options', $question)
                        ? $question['options']
                        : null,
                    'defaultQuestionObjectId' => array_key_exists('defaultQuestionObjectId', $question)
                        ? $question['defaultQuestionObjectId']
                        : null,
                    'type' => $question['type'],
                    'required' => $question['required'],
                    'formObjectId' => $form->objectId,
                ]);
                ++$i;
            }

            DB::commit();
            return response()->json([
                'message' => 'Formulario actualizado exitosamente.',
                'form' => Form::find($form->objectId)
            ], 201);

        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function update(Request $request) {

    }

    public function destroy(Request $request, $id) {
        try {
            DB::beginTransaction();
            $form = Form::where('isDeleted', false)
                ->where('businessObjectId', Auth::user()->businessObjectId)
                ->find($id);
            $form->isDeleted = true;
            $form->save();

            DB::commit();
            return response()->json([
                'message' => 'Formulario eliminado exitosamente.',
                'form' => $form
            ], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }
}
