<?php

namespace App\Http\Controllers;

use App\Custom\OperationCustom;
use App\Models\Operation;
use App\Models\PaymentMethod;
use App\Models\TattooArtist;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class OperationController extends Controller
{
    private function objectToArray($object) {
        return json_decode(json_encode($object), true);
    }
    public function index(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));
            $tattooArtists = $request->tattooArtists ? json_decode($request->tattooArtists) : [];
            $operations = [];

            for ($i = 0; $i < count($tattooArtists); ++$i) {
                $tattooArtist = TattooArtist::find($tattooArtists[$i]);
                if (!$tattooArtist) continue;

                $businessObjectId = $tattooArtist->studio->businessObjectId;
                if ($businessObjectId != Auth::user()->businessObjectId) continue;

                $operationCustom = new OperationCustom($request->startDate, $request->endDate, $tattooArtist->objectId);
                $currentOperations = $this->objectToArray($operationCustom->operations());
                $operations = array_merge($currentOperations, $operations);

            }

            $operationCustom = new OperationCustom($request->startDate, $request->endDate);
            $formattedOperations = $operationCustom->formatOperations($operations);
            return response()->json(['operations' => $formattedOperations], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function update(Request $request, $id) {
        try {
            DB::beginTransaction();

            $this->validate($request, $this->validator($request));

            $data = $request->all();
            $operation = Operation::find($id);

            $tattooArtist = $operation->tattooArtist;
            if (!$tattooArtist) return response()->json(['error' => 'Bad URI'], 404);

            $businessObjectId = $tattooArtist->studio->businessObjectId;
            if ($businessObjectId != Auth::user()->businessObjectId) {
                return response()->json(['error' => 'Bad URI'], 404);
            }

            $sale = $operation->sale;
            $advancePayment = $operation->advancePayment;

            if ($sale) {
                $salePercentage = $tattooArtist->salePercentage;

                $paymentMethod = PaymentMethod::where('type', 'Otro')->first();
                $data['salePercentage'] = $salePercentage;

                if ($paymentMethod->objectId != $request->paymentMethodObjectId) {
                    unset($data['otherPaymentMethod']);
                }

                $sale->update($data);
            }

            $data['studioObjectId'] = $tattooArtist->studioObjectId;
            $operation->update($data);

            DB::commit();

            return response()->json([
                'message' => 'Transacción actualizada exitosamente',
                'operation' => Operation::find($id)
            ], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function destroy($id)
    {
        try {
            $operation = Operation::find($id);
            if (!$operation) return response()->json(['error' => 'Bad URI'], 404);

            $tattooArtist = $operation->tattooArtist;
            $businessObjectId = $tattooArtist->studio->businessObjectId;
            if ($businessObjectId != Auth::user()->businessObjectId)
                return response()->json(['error' => 'Bad URI'], 404);

            $operation->deleteOperation();

            return response()->json(['message' => "Transacción eliminada exitosamente"], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    private function validator($request)
    {
        $data = $request->all();
        $rules = [
            'startDate' => 'required',
            'endDate' => 'required',
            'tattooArtists' => 'nullable',

            'amount' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
            'concept' => 'nullable',
            // 'salePercentage' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
            'clientName' => 'required',
            // 'clientEmail' => 'nullable|min:5|max:60|regex:/^.+@.+[.]+.+$/',
            'advance' => 'nullable|regex:/^(?!0\d)\d*(\.\d+)?$/',
            'tattooArtistObjectId' => 'required',
            'paymentMethodObjectId' => 'required',
            'otherPaymentMethod' => 'nullable',
            'date' => 'nullable'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
