<?php

namespace App\Http\Controllers;

use App\Custom\OperationCustom;
use App\Custom\PayrollCustom;
use App\Models\Business;
use App\Models\Payroll;
use App\Models\Studio;
use App\Models\TattooArtist;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PayrollController extends Controller
{
    public function previewIndex(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));
            $tattooArtists = $request->tattooArtists ? json_decode($request->tattooArtists) : [];
            $newTattooArtists = [];
            foreach ($tattooArtists as $id) {
                $tattooArtist = TattooArtist::find($id);
                if ($tattooArtist) {
                    $businessObjectId = $tattooArtist->studio->businessObjectId;
                    if ($businessObjectId != Auth::user()->businessObjectId) continue;
                }
                $operationCustom = new OperationCustom($request->startDate, $request->endDate, $tattooArtist->objectId);
                $isValid = PayrollCustom::validateDates([
                    'startDate' => $request->startDate,
                    'endDate' => $request->endDate,
                    'tattooArtistObjectId' => $tattooArtist->objectId
                ]);
                $tattooArtist['summary'] = $operationCustom->summary();
                $tattooArtist['validDates'] = $isValid;
                $newTattooArtists[] = $tattooArtist;
            }

            return response()->json(['payrollPreviews' => $newTattooArtists], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function show($id)
    {
        try {
            $payroll = Payroll::find($id);
            if (!$payroll) return response()->json(['error' => 'Bad URI'], 404);

            $studio = Studio::find($payroll->tattooArtist->studioObjectId);
            $businessObjectId = $studio->businessObjectId;
            if ($businessObjectId != Auth::user()->businessObjectId)
                return response()->json(['error' => 'Bad URI'], 404);

            $operationCustom = new OperationCustom($payroll->startDate, $payroll->endDate, $payroll->tattooArtistObjectId);
            $formattedOperations = $operationCustom->formatOperations($operationCustom->operations());

            return response()->json([
                'operations' => $formattedOperations,
                'payroll' => $payroll
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $this->validate($request, $this->validator($request));
            $tattooArtistsInfo = $request->tattooArtists ? json_decode($request->tattooArtists) : [];
            $results = [
                'success' => [],
                'errors' => []
            ];
            foreach ($tattooArtistsInfo as $tattooArtistInfo) {
                $isValid = PayrollCustom::validateDates([
                    'startDate' => $request->startDate,
                    'endDate' => $request->endDate,
                    'tattooArtistObjectId' => $tattooArtistInfo->tattooArtistObjectId
                ]);

                $tattooArtist = TattooArtist::find($tattooArtistInfo->tattooArtistObjectId);
                if (!$tattooArtist) continue;

                $businessObjectId = $tattooArtist->studio->businessObjectId;
                if ($businessObjectId != Auth::user()->businessObjectId)
                    continue;

                if (!$isValid) {
                    $results['errors'][] = [
                        'error' => 'Hay un conflicto con las fechas: ' . $tattooArtist->user->name
                    ];
                    continue;
                }

                $payroll = Payroll::create([
                    'startDate' => $request->startDate,
                    'endDate' => $request->endDate,
                    'tattooArtistObjectId' => $tattooArtist->objectId
                ]);

                // $recurringCharges = $tattooArtist->recurringCharges ? $tattooArtist->recurringCharges : [];
                // foreach ($recurringCharges as $recurringCharge) {
                //     $recurringCharge->collectPayments($payroll);
                // }

                if ($tattooArtistInfo->sendMail) {
                    PayrollCustom::sendPayroll($payroll);
                }

                $results['success'][] = [
                    'message' => 'Corte generado correctamente: ' . $tattooArtist->user->name
                ];
            }

            $response = ['results' => $results];
            $statusCode = 201;
            if (count($results['errors']) > 0) {
                $statusCode = 207;
                $response['error'] = 'Los cortes se han generado con errores';
            } else {
                $response['message'] = 'Los cortes se han generado correctamente';
            }

            DB::commit();
            return response()->json($response, $statusCode);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function sendTicket(Request $request, $id)
    {
        try {
            $payroll = Payroll::find($id);
            if (!$payroll) return response()->json(['error' => 'No se encontró el corte'], 404);

            PayrollCustom::sendPayroll($payroll);

            return response()->json([
                'message' => 'El reporte de corte ha sido enviado'
            ], 200);
        } catch (\Throwable $e) {
            return response()->json([
                'error' => 'Ocurrió un error al enviar el correo'
            ], 406);
        }
    }

    // public function update(Request $request, $id) {
    //     try {
    //         $studio = Studio::where([
    //             ['isDeleted', false],
    //             ['businessObjectId', Auth::user()->businessObjectId]
    //         ])->find($id);

    //         if (!$studio) return response()->json(['error' => 'Bad URI'], 404);

    //         $this->validate($request, $this->validator($request));

    //         $data = $request->all();
    //         unset($data['logo']);
    //         $studio->update($data);

    //         if ($request->hasFile('logo')) {
    //             $business = $studio->business;
    //             $path = "business/$business->objectId/studio";
    //             $studio->logo = $this->createFile($path, 'logo', $studio->objectId, $request->logo);
    //             $studio->save();
    //         }

    //         return response()->json(['studio' => Studio::find($studio->objectId)], 200);
    //     } catch (ModelNotFoundException $e) {
    //         return response()->json(['error' => 'No content 1'], 406);
    //     } catch (ValidationException $e) {
    //         return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
    //     } catch (\Throwable $e) {
    //         return response()->json(['error' => 'No content 2'], 406);
    //     }
    // }

    public function destroy($id)
    {
        try {
            $payroll = Payroll::find($id);
            if (!$payroll) return response()->json(['error' => 'Bad URI'], 404);

            $studio = Studio::find($payroll->tattooArtist->studioObjectId);
            $businessObjectId = $studio->businessObjectId;
            if ($businessObjectId != Auth::user()->businessObjectId)
                return response()->json(['error' => 'Bad URI'], 404);

            $payroll->delete();

            return response()->json(['message' => 'Corte eliminado exitosamente'], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    private function validator($request)
    {
        $data = $request->all();
        $rules = [
            'startDate' => 'required',
            'endDate' => 'required',
            'tattooArtists' => 'nullable',
            'tattooArtistObjectId' => 'required',
            'sendMail' => 'nullable'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
