<?php

namespace App\Http\Controllers;

use App\Models\Business;
use App\Models\Picture;
use App\Models\Product;
use App\Models\Studio;
use App\Traits\FilesTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProductController extends Controller
{
    use FilesTrait;
    public function getProductsForTattoo(Request $request) {
        try {
            $rules = [
                'perPage' => 'required|integer',
                'currentPage' => 'required|integer',
                'search' => 'sometimes|string',
                'orderBy' => 'required|string'
            ];
            $messages = [
                'perPage.required' => 'Los productos por página son requeridos.',
                'perPage.integer' => 'Los productos por página son deben ser un entero.',
                'currentPage.required' => 'La página actual es requerida.',
                'currentPage.integer' => 'La página actual debe ser un entero.',
                'search.string' => 'El filtro debe ser una cadena de texto.',
                'orderBy.required' => 'El tipo de ordenamiento es requerido.',
                'orderBy.string' => 'El tipo de ordenamiento debe ser una cadena de texto.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $studio = Studio::where('isDeleted', false)
                ->find(Auth::user()->studioObjectId);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 400);

            $business = Business::where('isDeleted', false)
                ->find($studio->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 400);

            $products = $this->getProducts($business, $request);

            return response()->json($products, 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }
    public function getProductsForAdmin(Request $request) {
        try {

            $rules = [
                'perPage' => 'required|integer',
                'currentPage' => 'required|integer',
                'search' => 'sometimes|string',
                'orderBy' => 'required|string'
            ];
            $messages = [
                'perPage.required' => 'Los productos por página son requeridos.',
                'perPage.integer' => 'Los productos por página son deben ser un entero.',
                'currentPage.required' => 'La página actual es requerida.',
                'currentPage.integer' => 'La página actual debe ser un entero.',
                'search.string' => 'El filtro debe ser una cadena de texto.',
                'orderBy.required' => 'El tipo de ordenamiento es requerido.',
                'orderBy.string' => 'El tipo de ordenamiento debe ser una cadena de texto.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $business = Business::where('isDeleted', false)
                ->find(Auth::user()->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 400);

            $products = $this->getProducts($business, $request);

            return response()->json($products, 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }
    public function getProducts($business, $request) {
            $pages = ceil(
                Product::where([['name', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                    ->orWhere([['price', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                    ->orWhere([['description', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                    ->count() / $request->perPage
            );

            $products = Product::where([['name', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                ->orWhere([['price', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                ->orWhere([['description', 'like', "%$request->search%"], ['isDeleted', false], ['businessObjectId', $business->objectId]])
                ->orderBy('name', $request->orderBy)
                ->skip($request->currentPage * $request->perPage)
                ->take($request->perPage)
                ->get();

            if ($products && count($products) > 0) {
                foreach($products as $key => $product) {
                    $products[$key] = $this->formatProduct($product);
                }
            }

            return [
                'products' => $products,
                'pages' => $pages
            ];
    }

    public function formatProduct($product) {
        $product = Product::find($product->objectId);
        $pictures = Picture::where('isDeleted', false)
            ->where('productObjectId', $product->objectId)
            ->get();

        $product['pictures'] = $pictures;

        return $product;
    }

    public function store(Request $request) {
        try {
            DB::beginTransaction();
            $rules = [
                'name' => 'required|string',
                'price' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
                'description' => 'sometimes|string',
                'pictures' => 'required|array|min:1|max:1',
                'pictures.*' => 'required|mimes:jpeg,bmp,png,jpg|max:5000'
            ];
            $messages = [
                'name.required' => 'El nombre es requerido.',
                'name.string' => 'El nombre debe ser una cadena de texto.',
                'price.required' => 'El precio es requerido.',
                'price.regex' => 'El formato del precio es incorrecto.',
                'description.string' => 'La descripción debe ser una cadena de texto.',
                'pictures.required' => 'El arreglo de imagenes es requerido.', 'pictures.array' => 'Las imagenes deben ser un arreglo.',
                'pictures.max' => 'El arreglo de imagenes no puede ser mayor a :max imagen(es).',
                'pictures.*.required' => 'Las imagenes del arreglo son requeridas.',
                'pictures.*.mimes' => 'Las imágenes deben ser del tipo (jpeg,bmp,png,jpg).',
                'pictures.*.max' => 'Las imagenes no pueden ser mayor a :max kb.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $business = Business::where('isDeleted', false)
                ->find(Auth::user()->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $product = Product::create([
                'name' => $request->name,
                'price' => $request->price,
                'description' => $request->description,
                'businessObjectId' => $business->objectId
            ]);

            $pictures = $request->pictures;
            if ($pictures) {
                foreach($pictures as $picture) {
                    $pic = Picture::create([
                        'path' => '',
                        'isDeleted' => true,
                        'productObjectId' => $product->objectId
                    ]);

                    $folder = "products/$product->objectId/pictures";
                    $path = $this->createFile($folder, 'picture', $pic->objectId, $picture);

                    $pic->path = $path;
                    $pic->isDeleted = false;
                    $pic->save();
                }
            }

            DB::commit();
            return response()->json([
                'message' => 'Producto creado exitosamente.',
            ], 201);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }

    public function update(Request $request, $id) {
        try {
            DB::beginTransaction();
            $rules = [
                'name' => 'required|string',
                'price' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
                'description' => 'sometimes|string',
                'pictures' => 'sometimes|array|max:1',
                'pictures.*' => 'required|mimes:jpeg,bmp,png,jpg|max:5000',
                'uploadedPictures' => 'sometimes|array|max:1',
                'uploadedPictures.*' => 'required|uuid',
            ];
            $messages = [
                'name.required' => 'El nombre es requerido.',
                'name.string' => 'El nombre debe ser una cadena de texto.',
                'price.required' => 'El precio es requerido.',
                'price.regex' => 'El formato del precio es incorrecto.',
                'description.string' => 'La descripción debe ser una cadena de texto.',
                // 'pictures.required' => 'El arreglo de imagenes es requerido.',
                'pictures.array' => 'Las imagenes deben ser un arreglo.',
                'pictures.max' => 'El arreglo de imagenes no puede ser mayor a :min imagen(es).',
                'pictures.*.required' => 'Las imagenes del arreglo son requeridas.',
                'pictures.*.mimes' => 'Las imágenes deben ser del tipo (jpeg,bmp,png,jpg).',
                'pictures.*.max' => 'Las imagenes no pueden ser mayor a :max kb.',
                // 'uploadedPictures.required' => 'El arreglo de imagenes es requerido.',
                'uploadedPictures.array' => 'Las imagenes deben ser un arreglo.',
                'uploadedPictures.*.required' => 'El id de la imagen es requerido.',
                'uploadedPictures.*.uuid' => 'El id de la imagen debe ser un uuid.',
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $business = Business::where('isDeleted', false)
                ->find(Auth::user()->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $product = Product::where('isDeleted', false)
                ->where('businessObjectId', $business->objectId)
                ->find($id);

            if (!$product) return response()->json([
                'error' => 'Producto no válido.'
            ], 404);


            $product->update([
                'name' => $request->name,
                'price' => $request->price,
                'description' => $request->description
            ]);

            if (!$request->pictures && !$request->uploadedPictures) {
                return response()->json([
                    'error' => 'Las imágenes son requeridas.'
                ], 400);
            }

            $totalImages = 0;
            if ($request->pictures)
                $totalImages = count($request->pictures);
            if ($request->uploadedPictures)
                $totalImages += count($request->uploadedPictures);

            if ($totalImages > 1) return response()->json([
                'error' => 'Puedes subir máximo 1 imagen'
            ], 400);

            if ($totalImages == 0) return response()->json([
                'error' => 'Agrega al menos una imagen.'
            ], 400);

            $delPictures = Picture::where('isDeleted', false)
                ->where('productObjectId', $product->objectId)
                ->get();

            $delPictures->each->update([
                'isDeleted' => true
            ]);
            $pictures = $request->pictures;
            if ($pictures) {


                foreach($pictures as $picture) {
                    $pic = Picture::create([
                        'path' => '',
                        'isDeleted' => true,
                        'productObjectId' => $product->objectId
                    ]);

                    $folder = "products/$product->objectId/pictures";
                    $path = $this->createFile($folder, 'picture', $pic->objectId, $picture);

                    $pic->path = $path;
                    $pic->isDeleted = false;
                    $pic->save();
                }
            }

            $uploadedPictures = $request->uploadedPictures;
            if ($uploadedPictures) {
                foreach($uploadedPictures as $picture) {
                    $pic = Picture::where('productObjectId', $product->objectId)
                            ->find($picture);

                    if (!$pic) return response()->json([
                        'error' => 'Imagen no válida.'
                    ], 400);

                    $pic->isDeleted = false;
                    $pic->save();
                }
            }

            DB::commit();
            return response()->json([
                'message' => 'Producto actualizado exitosamente.',
            ], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }

    public function destroy(Request $request, $id) {
        try {
            $product = Product::where('isDeleted', false)
                ->where('businessObjectId', Auth::user()->businessObjectId)
                ->find($id);

            if (!$product) return response()->json([
                'error' => 'Producto no válido.'
            ], 404);

            $product = $product->deleteProduct();

            return response()->json([
                'message' => 'Producto eliminado exitosamente.',
                'product' => $product
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }
}
