<?php

namespace App\Http\Controllers;

use App\Models\Business;
use App\Models\Operation;
use App\Models\Product;
use App\Models\ProductPurchase;
use App\Models\ProductPurchaseItem;
use App\Models\Sale;
use App\Models\Studio;
use App\Models\TattooArtist;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class ProductPurchaseController extends Controller
{
    public function index(Request $request) {
        try {
            $rules = [
                'startDate' => 'sometimes|date',
                'endDate' => 'sometimes|date',
            ];
            $messages = [
                'startDate.date' => 'Fecha de inicio incorrecta.',
                'endDate.date' => 'Fecha de fin incorrecta.'
            ];

            $validator = Validator::make($request->all(), $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $business = Business::where('isDeleted', false)
                ->find(Auth::user()->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 400);

            $endDate = $request->has('endDate')
                ? date('Y-m-d', strtotime($request->endDate))
                : date('Y-m-d');
            $startDate = $request->has('startDate')
                ? date('Y-m-d', strtotime($request->startDate))
                : date('Y-m-d', strtotime("$endDate - 7 days"));

            $productPurchases = ProductPurchase::where('businessObjectId', $business->objectId)
                ->where('isDeleted', false)
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->orderBy('created_at', 'DESC')
                ->get();

            $groupedProductPurchases = $this->groupBy($productPurchases);

            return response()->json([
                'productPurchases' => $groupedProductPurchases
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2', 'msg' => $e->getMessage()], 406);
        }
    }

    public function formatProductPurchase($productPurchase) {
        $productPurchase = ProductPurchase::find($productPurchase->objectId);
        $productPurchaseItems = ProductPurchaseItem::where('isDeleted', false)
            ->where('productPurchaseObjectId', $productPurchase->objectId)
            ->get();
        // $sale = Sale::with(['paymentMethod'])->find($productPurchase->saleObjectId);
        $tattooArtist = TattooArtist::find($productPurchase->tattooArtistObjectId);

        $productPurchase['productPurchaseItems'] = $productPurchaseItems;
        // $productPurchase['sale'] = $sale;
        $productPurchase['tattooArtist'] = $tattooArtist;

        return $productPurchase;
    }

    public function groupBy($data) {
        $result = [];
        if ($data && count($data) > 0) {
            $i = 0;
            foreach($data as $d) {
                $date = date('Y-m-d', strtotime($d->created_at));

                if (array_key_exists($i, $result)) {
                    if (array_key_exists($date, $result[$i])) {
                        $result[$i][$date][] = $this->formatProductPurchase($d);
                    } else {
                        ++$i;
                        $result[$i] = [];
                        $result[$i][$date] = [];
                        $result[$i][$date][] = $this->formatProductPurchase($d);
                        continue;
                    }
                } else {
                    $result[$i] = [];
                    $result[$i][$date] = [];
                    $result[$i][$date][] = $this->formatProductPurchase($d);
                }
            }
        }

        return $result ?? [];
    }

    public function store(Request $request) {
        try {
            DB::beginTransaction();
            $rules = [
                'saleObjectId' => 'required|uuid',
                'shoppingCart' => 'required|array|min:1',
                'shoppingCart.*.objectId' => 'required|uuid',
                'shoppingCart.*.quantity' => 'required|integer'
            ];
            $messages = [
                'saleObjectId.required' => 'El id de la venta es requerido.',
                'saleObjectId.uuid' => 'El id de la venta debe ser un uuid.',
                'shoppingCart.required' => 'Los productos son requeridos.',
                'shoppingCart.array' => 'Los productos deben ser un arreglo.',
                'shoppingCart.min' => 'Debes agregar al menos un producto.',
                'shoppingCart.*.objectId.required' => 'El id del producto es requerido.',
                'shoppingCart.*.objectId.uuid' => 'El id del producto debe ser un uuid.',
                'shoppingCart.*.quantity.required' => 'La cantidad es requerida.',
                'shoppingCart.*.quantity.integer' => 'La cantidad debe ser un entero.',
            ];

            $data = json_decode($request->all()['shoppingCart'], true);
            $validator = Validator::make([
                'shoppingCart' => $data,
                'saleObjectId' => $request->saleObjectId
            ], $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    'error'=> $validator->errors()->first()
                ], 400);
            }

            $studio = Studio::where('isDeleted', false)
                ->find(Auth::user()->studioObjectId);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 400);

            $business = Business::where('isDeleted', false)
                ->find($studio->businessObjectId);

            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 400);

            $sale = Sale::find($request->saleObjectId);

            if (!$sale) return response()->json([
                'error' => 'Venta no válida.'
            ], 400);

            $operation = Operation::where('saleObjectId', $sale->objectId)
                ->where('studioObjectId', $studio->objectId)
                ->first();

            if (!$operation) return response()->json([
                'error' => 'Venta no válida.'
            ], 400);

            $productPurchase = ProductPurchase::create([
                'saleObjectId' => $sale->objectId,
                'tattooArtistObjectId' => $sale->operation->tattooArtistObjectId,
                'clientName' => $sale->clientName,
                // 'clientEmail' => $sale->clientEmail,
                'paymentMethod' => $sale->paymentMethod->type == 'Otro'
                    ? $sale->otherPaymentMethod
                    : $sale->paymentMethod->type,
                'businessObjectId' => $business->objectId
            ]);

            foreach($data as $shoppingItem) {
                $product = Product::where('isDeleted', false)
                    ->where('businessObjectId', $business->objectId)
                    ->find($shoppingItem['objectId']);

                if (!$product) return response()->json([
                    'error' => 'Producto no válido.'
                ], 400);

                $productPurchaseItem = ProductPurchaseItem::create([
                    'name' => $product->name,
                    'description' => $product->description,
                    'price' => $product->price,
                    'quantity' => $shoppingItem['quantity'],
                    'productPurchaseObjectId' => $productPurchase->objectId,
                    'productObjectId' => $product->objectId
                ]);
            }

            DB::commit();
            return response()->json([
                'message' => 'Venta de productos generada correctamente.'
            ], 201);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2', 'data' => $e->getMessage()], 406);
        }
    }
}
