<?php

namespace App\Http\Controllers;

use App\Mail\RecoveryCodeMail;
use App\Mail\SendEmail;
use App\Models\Session;
use App\Models\User;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\ValidationException;

class SessionController extends Controller
{

    public function login(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));

            $user = User::where([
                ['email', $request->email],
                ['isDeleted', false]
            ])->first();

            if (!$user) return response()->json(['error' => 'Bad URI'], 404);

            if ($user && Hash::check($request->password, $user->password)) {

                $session = Session::create([
                    'token' => Str::random(150),
                    'expirationDate' => date('Y-m-d H:i:s', strtotime('now + 3 months')),
                    'userObjectId' => $user->objectId
                ]);

                $user = User::with(['admin', 'superAdmin', 'tattooArtist'])
                    ->where('objectId', $user->objectId)
                    ->first();

                $admin = $user->admin
                    ? $user->admin
                    : null;

                $tattooArtist = $user->tattooArtist;
                $studio = $tattooArtist ? $tattooArtist->studio : null;
                $business = $studio ? $studio->business : null;
                $admin = $business ? $business->admin : $admin;

                $isDisabled = $admin ? $admin->isDisabled : false;
                $expiredSubscription = false;

                $currentDate = time();
                $startDate = $admin ? strtotime($admin->startDate) : null;
                $endDate = $admin ? strtotime($admin->endDate) : null;
                if ($currentDate < $startDate || $currentDate > $endDate) {
                    $expiredSubscription = true;
                }

                if ($isDisabled) {
                    return response()->json([
                        'isDisabled' => $isDisabled,
                        'user' => $user,
                        'session' => $session,
                        'expiredSubscription' => $expiredSubscription,
                        'error' => 'No se pudo iniciar sesión'
                    ], 401);
                } else {
                    if ($tattooArtist && $expiredSubscription)
                        return response()->json([
                            'isDisabled' => $isDisabled,
                            'user' => $user,
                            'session' => $session,
                            'error' => 'No se pudo iniciar sesión'
                        ], 401);

                    return response()->json([
                        'isDisabled' => $isDisabled,
                        'user' => $user,
                        'session' => $session,
                        'message' => 'Bienvenido al sistema'
                    ], 200);
                }
            } else {
                return response()->json(['error' => 'Error de usuario y/o contraseña'], 400);
            }
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function logout(Request $request)
    {
        try {
            Session::where('token', $request->header('Authorization'))->first()->delete();

            return response()->json(['message' => 'Has cerrado sesión'], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function sendRecoveryCode(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));

            $user = User::where([
                ['email', $request->email],
                ['isDeleted', false]
            ])->first();
            if (!$user) return response()->json(['error' => 'Bad URI'], 404);

            $recoveryCode = Str::random(8);
            $user->recoveryCode = $recoveryCode;
            $user->save();

            $mail = new RecoveryCodeMail($user);
            Mail::to($request->email)->send($mail);

            return response()->json(['message' => 'El código ha sido enviado'], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function resetPassword(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));

            $user = User::where([
                ['email', $request->email],
                ['isDeleted', false]
            ])->first();
            if (!$user) return response()->json(['error' => 'Bad URI'], 404);

            if (!$user->recoveryCode) {
                return response()->json(['error' => 'El código de recuperación ha expirado'], 406);
            }

            if ($request->recoveryCode != $user->recoveryCode) {
                return response()->json(['error' => 'El código de recuperación es incorrecto'], 406);
            }

            $user->password = Hash::make($request->password);
            $user->recoveryCode = null;
            $user->save();

            return response()->json(['message' => 'Contraseña actualizada correctamente'], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $th) {
            return response()->json(['error' => $th->getMessage()], 406);
        }
    }


    private function validator($request)
    {
        $data = $request->all();
        $rules = [
            'email' => 'required|min:5|max:60|regex:/^.+@.+[.]+.+$/',
            'password' => 'required',
            'recoveryCode' => 'required|min:8'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
