<?php

namespace App\Http\Controllers;

use App\Custom\DashboardCustom;
use App\Models\Business;
use App\Models\Studio;
use App\Traits\FilesTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use SebastianBergmann\CodeCoverage\Report\Html\Dashboard;

class StudioController extends Controller
{
    use FilesTrait;

    public function index()
    {
        try {
            $studios = Studio::where([
                ['isDeleted', false],
                ['businessObjectId', Auth::user()->businessObjectId]
            ])->get();
            $plan = Auth::user()->plan;
            $limit = $plan
                ? $plan->studiosLimit
                : 0;

            return response()->json([
                'studios' => $studios,
                'limit' => $limit
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function show($id)
    {
        try {
            $studio = Studio::where([
                ['isDeleted', false],
                ['businessObjectId', Auth::user()->businessObjectId]
            ])->find($id);

            if (!$studio) return response()->json(['error' => 'Bad URI'], 404);

            return response()->json(['studio' => $studio], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function store(Request $request)
    {
        try {
            $this->validate($request, $this->validator($request));

            $plan = Auth::user()->plan;

            if ($plan) {
                $studiosLimit = $plan->studiosLimit;

                if ($studiosLimit != -1) {
                    $currentStudios = Studio::where([
                        ['businessObjectId', Auth::user()->businessObjectId],
                        ['isDeleted', false]
                    ])->count();

                    if ($currentStudios >= $studiosLimit) {
                        return response()->json([
                            'error' => 'Has alcanzado el número máximo de estudios para tu suscripción'
                        ], 406);
                    }
                }
            }

            $data = $request->all();
            $data['businessObjectId'] = Auth::user()->businessObjectId;
            unset($data['logo']);
            $studio = Studio::create($data);

            if ($request->hasFile('logo')) {
                $business = $studio->business;
                $path = "business/$business->objectId/studio";
                $studio->logo = $this->createFile($path, 'logo', $studio->objectId, $request->logo);
                $studio->save();
            }

            return response()->json(['message' => 'Sucursal creada exitosamente'], 201);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $studio = Studio::where([
                ['isDeleted', false],
                ['businessObjectId', Auth::user()->businessObjectId]
            ])->find($id);

            if (!$studio) return response()->json(['error' => 'Bad URI'], 404);

            $this->validate($request, $this->validator($request));

            $data = $request->all();
            unset($data['logo']);
            $studio->update($data);

            if ($request->hasFile('logo')) {
                $business = $studio->business;
                $path = "business/$business->objectId/studio";
                $studio->logo = $this->createFile($path, 'logo', $studio->objectId, $request->logo);
                $studio->save();
            }

            return response()->json([
                'message' => 'Estudio actualizado correctamente',
                'studio' => Studio::find($studio->objectId)
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function dashboard(Request $request, $id) {
        try {
            $business = Auth::user()->business;
            if (!$business) return response()->json([
                'error' => 'Negocio no válido.'
            ], 404);

            $studio = $business->studios()
                ->where('isDeleted', false)
                ->find($id);

            if (!$studio) return response()->json([
                'error' => 'Estudio no válido.'
            ], 404);

            return response()->json([
                'todayIncoming' => DashboardCustom::getTodayIncoming($id),
                'monthlyIncoming' => DashboardCustom::getMonthlyIncoming($id),
                'productMonthlyIncoming' => DashboardCustom::getProductMonthlyIncoming($id),
                'tattooArtistMonthlyIncoming' => DashboardCustom::getTattooArtistMonthlyIncoming($id)
            ], 200);

        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function destroy($id)
    {
        try {
            DB::beginTransaction();
            $studio = Studio::where([
                ['isDeleted', false],
                ['businessObjectId', Auth::user()->businessObjectId]
            ])->find($id);

            if (!$studio) return response()->json(['error' => 'Bad URI'], 404);

            $studio->deleteStudio();

            DB::commit();
            return response()->json(['message' => 'Estudio eliminado exitosamente'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    private function validator($request)
    {
        $max_file = env('UPLOAD_MAX_FILESIZE');

        $data = $request->all();
        $rules = [
            'name' => 'required',

            'street' => 'required|max:500',

            'phone' => 'required|digits:10|numeric',
            'website' => 'nullable',
            'email' => 'nullable|min:5|max:60|regex:/^.+@.+[.]+.+$/',
            'logo' => "nullable|max:$max_file",
            'legend' => 'nullable'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
