<?php

namespace App\Http\Controllers;

use App\Custom\MailCustom;
use App\Mail\SendEmail;
use App\Mail\SetPassword;
use App\Models\TattooArtist;
use App\Models\Business;
use App\Models\Form;
use App\Models\Studio;
use App\Models\User;
use App\Traits\FilesTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class TattooArtistController extends Controller
{

    use FilesTrait;

    public function index()
    {
        try {
            $business = Business::find(Auth::user()->businessObjectId);
            $studios = $business->studios;
            $tattooArtists = [];

            foreach ($studios as $studio) {
                $currentTattooArtists = TattooArtist::where([
                    ['isDeleted', false],
                    ['studioObjectId', $studio->objectId]
                ])->get();

                $tattooArtists = array_merge(
                    $tattooArtists,
                    json_decode(json_encode($currentTattooArtists), true)
                );
            }

            $plan = Auth::user()->plan;
            $limit = $plan
                ? $plan->tattooArtistsLimit
                : 0;

            return response()->json([
                'tattooArtists' => $tattooArtists,
                'limit' => $limit
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function tattooArtistsAllGuest(Request $request, $id)
    {
        try {
            $form = Form::where('isDeleted', false)
                ->find($id);

            if (!$form) return response()->json([
                'error' => 'Formulario no válido.'
            ], 404);

            $business = Business::where('isDeleted', false)
                ->find($form->businessObjectId);
            $studios = $business->studios;
            $tattooArtists = [];

            foreach ($studios as $studio) {
                $currentTattooArtists = TattooArtist::where([
                    ['studioObjectId', $studio->objectId],
                    ['isDeleted', false]
                ])->get();

                $tattooArtists = array_merge(
                    $tattooArtists,
                    json_decode(json_encode($currentTattooArtists), true)
                );
            }

            return response()->json([
                'tattooArtists' => $tattooArtists,
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function tattooArtistsAll(Request $request)
    {
        try {
            $business = Business::find(Auth::user()->businessObjectId);
            $studios = $business->studios;
            $tattooArtists = [];

            foreach ($studios as $studio) {
                $where = [
                    ['studioObjectId', $studio->objectId]
                ];
                if ($request->has('noDeleted'))
                    $where[] = [
                        'isDeleted', false
                    ];
                $currentTattooArtists = TattooArtist::where($where)->get();

                $tattooArtists = array_merge(
                    $tattooArtists,
                    json_decode(json_encode($currentTattooArtists), true)
                );
            }

            $plan = Auth::user()->plan;
            $limit = $plan
                ? $plan->tattooArtistsLimit
                : 0;

            return response()->json([
                'tattooArtists' => $tattooArtists,
                'limit' => $limit
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function tattooArtistsAsTattooArtist()
    {
        try {
            $studio = Studio::find(Auth::user()->studioObjectId);
            $tattooArtists = TattooArtist::where([
                ['isDeleted', false],
                ['studioObjectId', $studio->objectId]
            ])->get();

            return response()->json(['tattooArtists' => $tattooArtists], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function show($id)
    {
        try {
            $tattooArtist = TattooArtist::where('isDeleted', false)->find($id);
            return response()->json(['tattooArtist' => $tattooArtist], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $this->validate($request, $this->validator($request));

            $plan = Auth::user()->plan;

            if ($plan) {
                $tattooArtistsLimit = $plan->tattooArtistsLimit;

                if ($tattooArtistsLimit != -1) {
                    $currentStudios = Studio::where([
                        ['businessObjectId', Auth::user()->businessObjectId],
                        ['isDeleted', false]
                    ])->get();
                    $currentTattooArtists = 0;

                    foreach ($currentStudios as $studio) {
                        $currentTattooArtists += TattooArtist::where([
                            ['studioObjectId', $studio->objectId],
                            ['isDeleted', false]
                        ])->count();
                    }

                    if ($currentTattooArtists >= $tattooArtistsLimit) {
                        return response()->json([
                            'error' => 'Has alcanzado el número máximo de tatuadores para tu suscripción'
                        ], 406);
                    }
                }
            }

            $data = $request->all();
            unset($data['password']);
            unset($data['photo']);
            $data['password'] = Str::random(150);

            if (User::where([
                ['email', $data['email']],
                ['isDeleted', false]
            ])->count() > 0)
                return response()->json(['error' => 'El correo ya existe'], 406);

            $user = User::create($data);
            $tattooArtist = TattooArtist::create([
                'salePercentage' => $data['salePercentage'],
                'studioObjectId' => $data['studioObjectId'],
                'userObjectId' => $user->objectId,
            ]);

            if ($request->hasFile('photo')) {
                // $filename = $request->photo->getClientOriginalName();
                // $request->file('photo')->storeAs('/', $filename, 'public');
                $user->photo = $this->createFile('users', 'profilePhoto', $user->objectId, $request->photo);
                $user->save();
            }

            $recoveryCode = Str::random(8);
            $user->recoveryCode = $recoveryCode;
            $user->save();

            // $link = $user->setPasswordUrl;
            // $mail = new SendEmail("Ingresa a este enlace para establecer tu contraseña: $link", "Establecer contraseña");
            $mail = new SetPassword($user);
            Mail::to($request->email)->send($mail);

            DB::commit();

            return response()->json([
                'message' => 'Usuario creado correctamente',
                'tattooArtist' => TattooArtist::find($tattooArtist->objectId)
            ], 201);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            $this->validate($request, $this->validator($request));

            $tattooArtist = TattooArtist::where('isDeleted', false)->find($id);
            if (!$tattooArtist) return response()->json(['error' => 'Bad URI'], 404);
            $user = $tattooArtist->user;
            if (!$user) return response()->json(['error' => 'Bad URI'], 404);

            $data = $request->all();
            unset($data['password']);
            unset($data['photo']);

            $checkMail = User::where([
                ['email', $request->email],
                ['isDeleted', false]
            ])->first();
            if ($checkMail) {
                if ($checkMail->objectId != $user->objectId) {
                    return response()->json([
                        'error' => 'Ya existe una cuenta con este correo'
                    ], 406);
                }
            }

            $user->update($data);
            $tattooArtist->update($data);

            if ($request->hasFile('photo')) {
                // $filename = $request->photo->getClientOriginalName();
                // $request->file('photo')->storeAs('/', $filename, 'public');
                $user->photo = $this->createFile('users', 'profilePhoto', $user->objectId, $request->photo);
                $user->save();
            }

            DB::commit();

            return response()->json([
                'message' => 'Usuario actualizado correctamente',
                'tattooArtist' => TattooArtist::find($tattooArtist->objectId)
            ], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            DB::rollBack();
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function destroy($id)
    {
        try {
            DB::beginTransaction();

            $tattooArtist = TattooArtist::where('isDeleted', false)->find($id);
            if (!$tattooArtist) return response()->json(['error' => 'Bad URI'], 404);

            $tattooArtist->deleteTattooArtist();

            DB::commit();

            return response()->json(['message' => 'Tatuador eliminado exitosamente'], 200);
        } catch (ModelNotFoundException $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'No content 2'], 406);
        }
    }

    private function validator($request)
    {
        $max_file = env('UPLOAD_MAX_FILESIZE');

        $data = $request->all();
        $rules = [
            'name' => 'required',
            // 'email' => 'nullable|min:5|max:60|regex:/^.+@.+[.]+.+$/|unique:users',
            'photo' => "nullable|max:$max_file",
            'salePercentage' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
            'studioObjectId' => 'required'
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
