<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AdvancePayment;
use App\Models\Operation;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use App\Custom\PeriodicityCustom;
use App\Models\RecurringCharge;
use App\Models\TattooArtist;


class TattooArtistRecurringChargeController extends Controller
{
    public function index($id)
    {
        try {
            $tattooArtist = TattooArtist::find($id);
            $recurringCharges = [];

            if ($tattooArtist) {
                $businessObjectId = $tattooArtist->studio->businessObjectId;

                if (Auth::user()->businessObjectId == $businessObjectId)
                    $recurringCharges = $tattooArtist->recurringCharges;
            }

            return response()->json([
                'recurringCharges' => $recurringCharges
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    public function destroy(Request $request, $id)
    {
        try {
            $charge = RecurringCharge::find($id);
            $tattooArtist = $charge
                ? TattooArtist::where('isDeleted', false)->find($charge->tattooArtistObjectId)
                : null;

            $businessObjectId = $tattooArtist
                ? $tattooArtist->studio->businessObjectId
                : null;

            if ($businessObjectId == Auth::user()->businessObjectId) {
                $charge->delete();

                return response()->json([
                    'message' => 'El descuento recurrente ha sido eliminado'
                ], 200);
            }
        } catch (\Throwable $e) {
            return response()->json([
                'error' => 'Ocurrió un error'
            ], 406);
        }
    }

    public function store(Request $request, $id)
    {
        try {
            $this->validate($request, $this->validator($request));

            $valid = false;
            switch ($request->periodicity) {
                case PeriodicityCustom::MONTHLY:
                    $valid = ($request->day >= 1 && $request->day <= 31);
                    break;
                case PeriodicityCustom::WEEKLY:
                    $valid = ($request->day >= 1 && $request->day <= 7);
                    break;
            }

            if (!$valid)
                return response()->json([
                    'error' => 'No se pudo crear el cargo recurrente'
                ], 406);

            $tattooArtist = TattooArtist::find($id);
            $recurringCharge = null;
            if ($tattooArtist) {
                $businessObjectId = $tattooArtist->studio->businessObjectId;
                if (Auth::user()->businessObjectId == $businessObjectId)
                    $recurringCharge = RecurringCharge::create([
                        'periodicity' => $request->periodicity,
                        'day' => $request->day,
                        'amount' => $request->amount,
                        'concept' => $request->concept,
                        'tattooArtistObjectId' => $tattooArtist->objectId
                    ]);
            }

            return response()->json([
                'message' => 'Recargo recurrente creado correctamente',
                'recurringCharge' => $recurringCharge
            ], 201);
        } catch (ModelNotFoundException $e) {
            return response()->json(['error' => 'No content 1'], 406);
        } catch (ValidationException $e) {
            return response()->json(['error' => 'Bad Request', 'errors' => $e->errors()], 400);
        } catch (\Throwable $e) {
            return response()->json(['error' => $e->getMessage()], 406);
        }
    }

    private function validator($request)
    {
        $data = $request->all();
        $rules = [
            'amount' => 'required|regex:/^(?!0\d)\d*(\.\d+)?$/',
            'concept' => 'required',
            'day' => 'required',
            'periodicity' => 'required',
        ];

        $toValidate = [];
        foreach ($rules as $field => $rule) {
            if (array_key_exists($field, $data)) {
                $toValidate[$field] = $rule;
            }
        }

        return $toValidate;
    }
}
