<?php

namespace App\Models;

use App\Custom\PeriodicityCustom;
use App\Traits\UsesUuid;
use Illuminate\Database\Eloquent\Model;

class RecurringCharge extends Model
{
    use UsesUuid;
    /**
     * The attributes that aren't mass assignable.
     *
     * @var array
     */
    protected $guarded = ['objectId'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'periodicity',
        'day',
        'amount',
        'concept',
        'tattooArtistObjectId'
    ];

    /* The model's default values for attributes. @var array*/
    protected $attributes = [];

    protected $with = [];

    protected $appends = [
        'next_payment_date'
    ];

    public function tattooArtist()
    {
        return $this->belongsTo(TattooArtist::class, 'tattooArtistObjectId');
    }

    public function getNextPaymentDateAttribute()
    {
        $now = time();

        $day = null;
        $date = null;
        $dayToCompare = $this->day;
        switch ($this->periodicity) {
            case PeriodicityCustom::MONTHLY:
                $day = date('j', $now);
                $monthDays = date('t', $now);
                $nextMonthDays = date('t', strtotime('next month'));

                if ($this->day > $monthDays) {
                    $dayToCompare = $monthDays;
                }

                $diff = $dayToCompare - $day;

                if ($diff > 0) {
                    // en este mismo mes
                    $date = date('Y-m-d', strtotime("today + $diff days"));
                } else {
                    // el mes que viene
                    if ($this->day > $nextMonthDays) {
                        $dayToCompare = $nextMonthDays;
                    }

                    $dayToCompare -= 1;

                    $date = date('Y-m-d', strtotime("next month + $dayToCompare days"));
                }

                break;
            case PeriodicityCustom::WEEKLY:
                $day = date('N', $now);

                $day = PeriodicityCustom::WEEK_DAYS[$dayToCompare];
                $date = date('Y-m-d', strtotime("next $day"));
                break;
            default:
                return;
        }

        return $date;
    }

    public function collectPayment($date)
    {
        $advancePaymentData = [
            'amount' => $this->amount,
            'concept' => $this->concept,
            'tattooArtistObjectId' => $this->tattooArtistObjectId,
            'studioObjectId' => $this->tattooArtist->studioObjectId
        ];

        $day = null;
        $dayToCompare = $this->day;
        switch ($this->periodicity) {
            case PeriodicityCustom::MONTHLY:
                $day = date('j', $date);
                $monthDays = date('t', $date);

                if ($this->day > $monthDays) {
                    $dayToCompare = $monthDays;
                }
                break;
            case PeriodicityCustom::WEEKLY:
                $day = date('N', $date);
                break;
            default:
                return;
        }

        if ($day == $dayToCompare) {
            $advancePayment = AdvancePayment::create(['isRecurring' => true]);
            $advancePaymentData['date'] = date('Y-m-d H:i:s', $date);
            $advancePaymentData['advancePaymentObjectId'] = $advancePayment->objectId;
            $operation = Operation::create($advancePaymentData);
        }
    }

    public function collectPayments($payroll)
    {
        if (!$payroll) return;
        $startDate = strtotime($payroll->startDate);
        $endDate = strtotime($payroll->endDate);
        $dayInSeconds = 60 * 60 * 24;

        $advancePaymentData = [
            'amount' => $this->amount,
            'concept' => $this->concept,
            'tattooArtistObjectId' => $this->tattooArtistObjectId,
            'studioObjectId' => $this->tattooArtist->studioObjectId
        ];

        for ($currentDate = $startDate; $currentDate <= $endDate; $currentDate += $dayInSeconds) {
            $day = null;
            $dayToCompare = $this->day;
            switch ($this->periodicity) {
                case PeriodicityCustom::MONTHLY:
                    $day = date('j', $currentDate);
                    $monthDays = date('t', $currentDate);

                    if ($this->day > $monthDays) {
                        $dayToCompare = $monthDays;
                    }
                    break;
                case PeriodicityCustom::WEEKLY:
                    $day = date('N', $currentDate);
                    break;
                default:
                    return;
            }

            if ($day == $dayToCompare) {
                $advancePayment = AdvancePayment::create(['isRecurring' => true]);
                $advancePaymentData['date'] = date('Y-m-d H:i:s', $currentDate);
                $advancePaymentData['advancePaymentObjectId'] = $advancePayment->objectId;
                $operation = Operation::create($advancePaymentData);
            }
        }
    }
}
